<?php

namespace App\Http\Controllers;

use App\Http\Requests\StoreSaidaCaixaRequest;
use App\Models\CentroCusto;
use App\Models\SaidaCaixa;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class SaidaCaixaController extends Controller
{
    //  Mostra a view da listagem dos promitentes compradores e Cliente
    public function index()
    {

        // Recupera todos os registros da BD Saidas de Caixa
        $saidas = SaidaCaixa::orderBy('id', 'desc')->with(['centroCusto', 'user'])->paginate(6);
        $ultimoregisto = SaidaCaixa::orderBy('id', 'desc')->first();

        // Recupera todos os centros de custos
        $centrosCusto = CentroCusto::all();

        // Recupera todos os utilizadores
        $users = User::all();

        return view('Administracao.SaidaCaixa', compact('saidas', 'centrosCusto', 'users', 'ultimoregisto'));

    }

    public function store(StoreSaidaCaixaRequest $request)
    {

        // Validar formulario de cadastro das Saidas de Caixa
        $request->validated();

        // dd( $request);

        // Cadastrando a saida de caixa no banco de Dados
        $saidaCaixa = new SaidaCaixa;

        $saidaCaixa->referencia = $request->referencia;
        $saidaCaixa->data = $request->data;
        $saidaCaixa->descritivo = $request->descritivo;
        $saidaCaixa->observacao = $request->observacao;
        $saidaCaixa->valor = $request->valor;
        $saidaCaixa->centros_custo_id = $request->centros_custo_id;
        $saidaCaixa->users_id = $request->users_id;
        $saidaCaixa->recebido_por = $request->recebido_por;
        $saidaCaixa->status = 'on';  // Status on - Validação da data da saida de caixa
        $saidaCaixa->assinatura = $request->assinatura;

        // UPLOAD DOCUMENTO DO SUPORTE DOCUMENTAL

        if ($request->hasFile('suporte_documental') && $request->file('suporte_documental')->isValid()) {

            $requestDoc = $request->suporte_documental; // Endereço do documento
            $extensaoDoc = $requestDoc->extension(); // Extensão dO DOCUMENTO

            $novoNomeDoc = md5($requestDoc->getClientOriginalName().strtotime('now')).'.'.$extensaoDoc; // Novo nome do doc no sistema
            $requestDoc->move(public_path('Doc/ComprovativoSaidaCaixa'), $novoNomeDoc); // Move pra para de imagens do sistema

            $saidaCaixa->suporte_documental = $novoNomeDoc;
        } else {

            $saidaCaixa->suporte_documental = '';
        }

        // dd($saidaCaixa);
        $saidaCaixa->save();

        $ultimoregisto = SaidaCaixa::orderBy('id', 'desc')->first();

        return redirect()->route('fluxoCaixa', compact('ultimoregisto'))->with('msg', 'Saída de Caixa cadastrada com sucesso!');

    }

    public function show($id)
    {
        $saidaCaixa = SaidaCaixa::with(['centroCusto', 'user'])->findOrFail($id);

        // dd($saidaCaixa);
        return response()->json($saidaCaixa);

    }

    public function print($id)
    {
        $dado = SaidaCaixa::with(['centroCusto', 'user'])->findOrFail($id);

        // dd($saidaCaixa);

        return view('Administracao.ComponentesAdmin.ModelosImpressao.Cpt_Mi_ReciboSC', compact('dado'));
    }

    /**
     * Search saidas de caixa with parametrized filters
     */
    public function search(Request $request)
    {
        // Start with a base query
        $query = SaidaCaixa::orderBy('id', 'desc')->with(['centroCusto', 'user']);

        // Apply date range filter if provided
        if ($request->filled('start_date')) {
            $query->whereDate('data', '>=', $request->start_date);
        }

        if ($request->filled('end_date')) {
            $query->whereDate('data', '<=', $request->end_date);
        }

        // Apply centro de custo filter if provided
        if ($request->filled('centro_custo')) {
            $query->where('centros_custo_id', $request->centro_custo);
        }

        // Apply valor range filter if provided
        if ($request->filled('min_valor')) {
            $query->where('valor', '>=', $request->min_valor);
        }

        if ($request->filled('max_valor')) {
            $query->where('valor', '<=', $request->max_valor);
        }

        // Apply referencia filter if provided
        if ($request->filled('referencia')) {
            $query->where('referencia', 'LIKE', '%'.$request->referencia.'%');
        }

        // Apply descritivo filter if provided
        if ($request->filled('descritivo')) {
            $query->where('descritivo', 'LIKE', '%'.$request->descritivo.'%');
        }

        // Execute the query with pagination
        $saidas = $query->paginate(6)->withQueryString();
        $ultimoregisto = SaidaCaixa::orderBy('id', 'desc')->first();

        // Get other required data for the view
        $centrosCusto = CentroCusto::all();
        $users = User::all();

        // Return the view with the search results
        return view('Administracao.SaidaCaixa', compact('saidas', 'centrosCusto', 'users', 'ultimoregisto'));
    }
    
    /**
     * Atualiza os dados de uma saída de caixa existente
     */
    public function update(Request $request, $id)
    {
        // Validação dos campos
        $request->validate([
            'referencia' => 'required|string|max:50',
            'data' => 'required|date',
            'descritivo' => 'required|string|max:255',
            'observacao' => 'nullable|string|max:255',
            'valor' => 'required|numeric|min:0',
            'centros_custo_id' => 'required|exists:centros_custo,id',
            'users_id' => 'required|exists:users,id',
            'recebido_por' => 'required|string|max:50',
            'suporte_documental' => 'nullable|file|mimes:pdf,jpg,jpeg,png|max:2048',
        ]);
        
        // Buscar a saída de caixa
        $saidaCaixa = SaidaCaixa::findOrFail($id);
        
        // Atualizar os campos
        $saidaCaixa->referencia = $request->referencia;
        $saidaCaixa->data = $request->data;
        $saidaCaixa->descritivo = $request->descritivo;
        $saidaCaixa->observacao = $request->observacao;
        $saidaCaixa->valor = $request->valor;
        $saidaCaixa->centros_custo_id = $request->centros_custo_id;
        $saidaCaixa->users_id = $request->users_id;
        $saidaCaixa->recebido_por = $request->recebido_por;
        
        // Processar assinatura (se fornecida)
        if ($request->filled('assinatura')) {
            $saidaCaixa->assinatura = $request->assinatura;
        }
        
        // Processar suporte documental (se enviado)
        if ($request->hasFile('suporte_documental') && $request->file('suporte_documental')->isValid()) {
            $requestDoc = $request->suporte_documental;
            $extensaoDoc = $requestDoc->extension();
            
            $novoNomeDoc = md5($requestDoc->getClientOriginalName().strtotime('now')).'.'.$extensaoDoc;
            $requestDoc->move(public_path('Doc/ComprovativoSaidaCaixa'), $novoNomeDoc);
            
            // Apagar arquivo antigo se existir
            if (!empty($saidaCaixa->suporte_documental)) {
                $caminhoArquivoAntigo = public_path('Doc/ComprovativoSaidaCaixa/'.$saidaCaixa->suporte_documental);
                if (file_exists($caminhoArquivoAntigo)) {
                    unlink($caminhoArquivoAntigo);
                }
            }
            
            $saidaCaixa->suporte_documental = $novoNomeDoc;
        }
        
        // Salvar as alterações
        $saidaCaixa->save();
        
        return redirect()->route('fluxoCaixa')->with('msg', 'Saída de Caixa atualizada com sucesso!');
    }
}
