<?php

namespace App\Http\Requests;

use App\Models\User;
use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Validation\Rule;

class UserRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     */
    public function authorize(): bool
    {
        return true;
    }

    /**
     * Get the validation rules that apply to the request.
     *
     * @return array<string, \Illuminate\Contracts\Validation\ValidationRule|array|string>
     */
    public function rules(): array
    {
        $userId = $this->route('user');
        
        // Regras de validação
        $rules = [
            'name' => 'required|string|max:255',
            'nID' => ['required', 'string', 'max:50', 
                      Rule::unique('pessoais', 'nID')->ignore($userId ? User::find($userId)->id_pessoa ?? null : null)],
            'tipoID' => 'required|string|in:BI,Passaporte,Outro',
            'genero' => 'required|string|in:Masculino,Feminino,Outro',
            'dataNasc' => 'required|date',
            'tipoConta' => 'required|string|in:Administrador,Operador,Cliente',
            'contactoPrimario' => 'required|string|max:255',
            'email' => ['required', 'string', 'email', 'max:255', 
                      Rule::unique('users')->ignore($userId)],
            'telefone' => ['required', 'string', 'max:20', 
                          Rule::unique('pessoais', 'telefone')->ignore($userId ? User::find($userId)->id_pessoa ?? null : null)],
            'obs' => 'nullable|string',
        ];
        
        // Se for um novo registro ou se a senha foi fornecida para atualização
        if (!$userId || $this->filled('password')) {
            $rules['password'] = 'required|string|min:8|confirmed';
        }
        
        return $rules;
    }
    
    /**
     * Get custom messages for validator errors.
     *
     * @return array
     */
    public function messages(): array
    {
        return [
            'name.required' => 'O nome é obrigatório',
            'nID.required' => 'O número de identificação é obrigatório',
            'nID.unique' => 'Este número de identificação já está em uso',
            'tipoID.required' => 'O tipo de identificação é obrigatório',
            'genero.required' => 'O género é obrigatório',
            'dataNasc.required' => 'A data de nascimento é obrigatória',
            'tipoConta.required' => 'O tipo de conta é obrigatório',
            'contactoPrimario.required' => 'O contacto primário é obrigatório',
            'email.required' => 'O e-mail é obrigatório',
            'email.email' => 'O e-mail deve ser um endereço válido',
            'email.unique' => 'Este e-mail já está em uso',
            'telefone.required' => 'O telefone é obrigatório',
            'telefone.unique' => 'Este telefone já está em uso',
            'password.required' => 'A senha é obrigatória',
            'password.min' => 'A senha deve ter pelo menos 8 caracteres',
            'password.confirmed' => 'A confirmação da senha não corresponde',
        ];
    }
}
